﻿/*
	VERSION:		1.3
	
	DESCRIPTION:
	Pros:		A single image can be used,  and can contain many poses.
				You can specify the:  size,  number of poses,  number animation frames,  number of directions.
				Animation speed can be controlled.
				Animation can occelate back & forth,  or loop.
	Cons:	All poses must have the same number of animation frames.
				
	DETAILED DESCRIPTION:
	This sprite system places the character's feet at the origin point.	(Feet:  1/2 image width, full image height)
	
	USAGE:
	mySprite = makeSprite( this, "marle1_mc" );			// minimum usage
	makeSprite( this, "marle1_mc", "mySprite_mc", 10, 24, 32, 3, 4, 4, 2, 3 );			// maximum usage		(rm2k settings)
	
	PUBLIC FUNCTIONS:
	setDirection	Changes the direction.
	setPose			Changes the pose, and the animation mode.  (yoyo / repeat)
	setAnim			Turns animation on / off  (boolean)
	setSpeed		Changes the animation speed.
	setFrame		Shows the specified frame of animation.
	setCharset		Changes the charset image
	
	
	PUBLIC SETTINGS:
	direction		The direction that the sprite is facing.
	pose				The pose that the sprite is in.
	anim				Whether or not the sprite is animating.
	animYoyo		Whether the animation loops, or goes back & forth.
	animFrame	Which frame of animation is currently shown.
	animSpeed	How fast the sprite animates.		(How many frames to wait between animation frames)
	charset			The charset image that this sprite is using.
*/



function makeSprite( target, newCharset, newName, depth, spriteWidth, spriteHeight, maxAnimFrames, maxDirections, maxPoseColumns, maxPoseRows, newAnimSpeed )
{
	// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - 
	// CREATE
	// // This Sprite movieClip
	var newName = (newName) ? newName : String("sprite_"+Math.round(Math.random() * 99999) );		// default:	sprite_94627
	var depth = (depth) ? depth : 1;		// default depth = 1
	var _this = target.createEmptyMovieClip( newName, depth );
	// // image_mc
	_this.attachMovie( newCharset, "image_mc", 1);
	// // mask_mc
	_this.createEmptyMovieClip( "mask_mc", 2);
	_this.mask_mc.beginFill( 0x000000, 25);
	_this.mask_mc.moveTo(0, 0);		// start
	_this.mask_mc.lineTo(16, 0);		// right
	_this.mask_mc.lineTo(16, 16);	// down
	_this.mask_mc.lineTo(0, 16);		// left
	_this.mask_mc.lineTo(0, 0);		// up
	_this.mask_mc.endFill();					// stop
	_this.image_mc.setMask( _this.mask_mc );
	
	
	// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - 
	// PUBLIC FUNCTIONS
	_this.setDirection = function( newDirection )
	{
		if ( isNaN(newDirection) )
		{
			switch (newDirection)
			{
				case "up":
					newDirection = 0;
				break;
				case "right":
					newDirection = 1;
				break;
				case "down":
					newDirection = 2;
				break;
				case "left":
					newDirection = 3;
				break;
			}// switch
		}// if isNaN
		
		_this.direction = newDirection;
	}// setDirection()
	
	
	
	_this.setPose = function( newPose, newAnimYoyo )
	{
		_this.pose = newPose;
		
		_this.animYoyo = (newAnimYoyo) ? newAnimYoyo : _this.animYoyo;		// use new setting if specified.
	}// setPose
	
	
	
	_this.setAnim = function( isAnimating )
	{
		_this.anim = isAnimating;
	}// setAnim
	
	
	
	_this.setSpeed = function( newSpeed )
	{
		newSpeed = Math.floor(newSpeed);		// removes decimals
		if (newSpeed <= 0  ||  newSpeed == undefined){ newSpeed = 1 };		//	enforce a valid range.	(prevents divide by zero)
		_this.animSpeed = newSpeed;
	}// setSpeed()
	
	
	
	_this.setFrame = function( newFrame )
	{
		_this.animFrame = newFrame;
	}// setFrame()
	
	
	
	_this.setCharset = function( newCharset )
	{
		_this.charset = newCharset;
	}// setCharset()
	
	
	
	// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - 
	// SUPPORT FUNCTIONS
	
	_this.updateDirection = function()
	{
		_this.direction %= _this.maxDirections;		// Limit direction changes to be within bounds.
		_this.y -= _this.direction * _this.spriteHeight;
	}// updateDirection()
	
	
	
	_this.updatePose = function()
	{
		var maxPoses = _this.maxPoseColumns * _this.maxPoseRows;
		_this.pose %= maxPoses;		// Limit poses to be within bounds.
		
		var thisColumn = _this.pose % _this.maxPoseColumns;
		var columnWidth = _this.spriteWidth * _this.maxAnimFrames;
		var thisRow = Math.floor(_this.pose / _this.maxPoseColumns);
		var rowHeight = _this.spriteHeight * _this.maxDirections;
		_this.x -= (thisColumn * columnWidth);
		_this.y -= (thisRow * rowHeight);
	}// updateDirection()
	
	
	
	_this.updateAnimation = function()
	{
		if (_this.animYoyo)
		{	// animate back & forth
			if (_this.animFrame >= _this.maxAnimFrames-1)
			{	// if this is the last frame
				_this.animDirection = -1;		// animate backwards
			}// if	>= maxAnimFrames
			else if (_this.animFrame <= 0)
			{	// if this is the first frame
				_this.animDirection = 1;		// animate forwards
			}// if	<= 0
		}// if yoyo
		else
		{	// animate forward only
			_this.animFrame %= (_this.maxAnimFrames);
			_this.animDirection = 1;		// animate forwards
		}// if !yoyo
		
		_this.x -= _this.animFrame * _this.spriteWidth;
		
		_this.animFrame += _this.animDirection;
	}// updateDirection()
	
	
	
	_this.setupMask = function()
	{
		_this.image_mc.setMask( _this.mask_mc );
		_this.mask_mc._width = _this.spriteWidth;
		_this.mask_mc._height = _this.spriteHeight;
		_this.mask_mc._x = _this.imageOffsetX;
		_this.mask_mc._y = _this.imageOffsetY;
	}// setupMask()
	
	
	
	_this.setupCharset = function()
	{
		if(_this.charset != _this.oldCharset)
		{
			// erase image_mc in case it already exists
			_this.image_mc.removeMovieClip();
			// image_mc
			_this.attachMovie( _this.charset, "image_mc", 1, {_x:9999,_y:9999});		// place it out-of-sight until the mask is applied
			// remember in case the setting changes
			_this.oldCharset = _this.charset;
		}
	}// setupCharset()
	
	
	
	// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - 
	//  SETTINGS
	
	// Default charset settings
	_this.spriteWidth = (spriteWidth) ? spriteWidth : 16;		// default:		16 px wide
	_this.spriteHeight = (spriteHeight) ? spriteHeight : 16;		// default:		16 px tall
	_this.maxAnimFrames = (maxAnimFrames) ? maxAnimFrames : 1;		// default:		1 frame of animation
	_this.maxDirections = (maxDirections) ? maxDirections : 1;		// default:		1 direction
	_this.maxPoseColumns = (maxPoseColumns) ? maxPoseColumns : 1;		// default:		1 pose horizontally
	_this.maxPoseRows = (maxPoseRows) ? maxPoseRows : 1;		// default:		1 pose vertically
	_this.setCharset(newCharset);
	_this.animDirection = 1;
	_this.setSpeed(newAnimSpeed);
	
	// // Initialize Private Variables
	_this.x 								= 0;				// This offsets the sprite image horizontally.
	_this.y 								= 0;				// This offsets the sprite image vertically.
	_this.imageOffsetX			= -1*(spriteWidth/2);				// -12		// This offsets the image to place the registration point in the middle.
	_this.imageOffsetY			= -1*(spriteHeight/2) - 8;		// -24		// This offsets the image to place the registration point 8 pixels above the feet.
	_this.animDelay					= 0;				// This counter delays the animation until it hits animSpeed, then loops.
	
	// // Default Public Settings
	_this.direction		= 	0;
	_this.pose				= 	0;
	_this.anim 				= true;
	_this.animYoyo		= true;
	if (maxAnimFrames > 1){
		_this.animFrame 	= 1;
	}else{
		_this.animFrame 	= 0;
	}
	
	
	
	// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - 
	// LOOP
	_this.onEnterFrame = function()
	{
		// Set the charset
		_this.setupCharset();
		
		// Setup the Mask		(this loops in case settings change)
		_this.setupMask();
		
		// Do delay
		if (_this.animDelay == 0)
		{
			// Reset
			_this.x = 0;
			_this.y = 0;
			
			// Set direction
			_this.updateDirection();
			
			// Set pose
			_this.updatePose();
			
			// Toggle animation		(on / off)
			if (!_this.anim   &&   _this.maxAnimFrames > 1)
			{
				if (_this.maxAnimFrames > 1){
					_this.animFrame 	= 1;
				}else{
					_this.animFrame 	= 0;
				}
			}// if anim
			
			// Do animation
			_this.updateAnimation();
			
			
			// Set the image
			_this.image_mc._x = _this.x + _this.imageOffsetX;
			_this.image_mc._y = _this.y + _this.imageOffsetY;
		
		}// if:		animDelay = 0
		_this.animDelay++;
		_this.animDelay %= _this.animSpeed;
	}// onEnterFrame()
	
	
	
	// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - 
	// OUTPUT
	return _this;
	// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - 
}// makeSprite()